package examples.interpreter.java;

/* -*- Mode: Java; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * This file is part of the design patterns project at UBC
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * either http://www.mozilla.org/MPL/ or http://aspectj.org/MPL/.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is ca.ubc.cs.spl.patterns.
 *
 * Contributor(s):   
 */

/**
 * Implements the driver for the Intepreter design pattern example.<p> 
 *
 * Intent: <i>Given a language, defeine a representation fro its grammar along
 * with an interpreter that uses the representation to interpret sentences
 * in the language.</i><p>
 *
 * Participating objects are <code>BooleanContant</code>, <code>VariableExp
 * </code>, <code>OrExp</code>, <code>AndExp</code>, and <code>NotExp</code>
 * as <i>Expressions</i>. The <i>AbstractExpression</i> interface is defined 
 * in <code>BooelanExp</i>.<p>
 *
 * This example implements an interpreter for a language of boolean 
 * expressions. As a sample expression, "((true & x) | (y & !x))" is
 * interpreted for all possible boolean values for x and y. After that, 
 * y is replaced by another expression and the whole expression is
 * evaluated again. 
 *
 * <p><i>This is the Java version.</i><p>
 *
 * @author  Jan Hannemann
 * @author  Gregor Kiczales
 * @version 1.0, 05/13/02
 * 
 * @see BooleanExp
 */

public class Main {  
    
    /** 
     * Assigns boolean values to two <code>VariableExp</code>s and evaluates
     * an expression in the given context.
     *
     * @param x a boolean variable expression
     * @param xValue the value to assign to x
     * @param y another boolean variable expression
     * @param yValue the value to assign to y
     * @param context the context to evaluate the expression in
     * @param exp the expression to evaluate
     */

	private static void assignAndEvaluate(  VariableExp x, boolean xValue,
						VariableExp y, boolean yValue,
						Context context, BooleanExp exp) {
		context.assign(x, xValue);
		context.assign(y, yValue);
		boolean result = exp.evaluate(context);
		System.out.println("The result for (x="+xValue+", y="+yValue+") is: "+result);
	}

    /**
     * Implements the driver for the Intepreter design pattern example.<p> 
     *
     * This example implements an interpreter for a language of boolean 
     * expressions. As a sample expression, "((true & x) | (y & !x))" is
     * interpreted for all possible boolean values for x and y. After that, 
     * y is replaced by another expression and the whole expression is
     * evaluated again.   
     *
     * @args command-line parameters, unused.
     */

	public static void main(String[] args) {
		BooleanExp exp = null;
		Context context = new Context();
		
		VariableExp x = new VariableExp("X");
		VariableExp y = new VariableExp("Y");		
		
		exp = new OrExp(new AndExp(new BooleanConstant(true), x), 
					    new AndExp(y, new NotExp(x)));
		
		System.out.println("Testing Expr: ((true & x) | (y & !x))");			 

		assignAndEvaluate(x, false, y, false, context, exp);
		assignAndEvaluate(x, false, y, true,  context, exp);
		assignAndEvaluate(x, true,  y, false, context, exp);
		assignAndEvaluate(x, true,  y, true,  context, exp);
		
		VariableExp z = new VariableExp("Z");
		NotExp   notZ = new NotExp(z);
		
		BooleanExp replacement = exp.replace("Y", notZ);
		context.assign(z, false);
		boolean result = replacement.evaluate(context);

		System.out.println("The result for the replacement is: "+result);
	}
}
		