package examples.observer.aspectj;

/* -*- Mode: Java; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * This file is part of the design patterns project at UBC
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * either http://www.mozilla.org/MPL/ or http://aspectj.org/MPL/.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is ca.ubc.cs.spl.patterns.
 *
 * Contributor(s):   
 */

import ca.ubc.cs.spl.pattern.library.ObserverProtocol;

/**
 * Concretizes the observing relationship for <code>Screen</code> (subject) 
 * and <code>Screen</code> (observer). Calls of <code>Screen.display(String)
 * </code> trigger updates.
 *
 * @author  Gregor Kiczales
 * @author  Jan Hannemann
 * @version 1.0, 05/13/02
 */

public aspect ScreenObserver extends ObserverProtocol{

  /**
   * Assings the <code>Subject</code> role to the <code>Screen</code> class.
   * Roles are modeled as (empty) interfaces.
   */
   
  declare parents: Screen implements Subject;

  /**
   * Assings the <code>Observer</code> role to the <code>Screen</code> class.
   * Roles are modeled as (empty) interfaces.
   */
   
  declare parents: Screen implements Observer;

  /**
   * Specifies the joinpoints that constitute a change to the
   * <code>Subject</code>. Captures calls to <code>Screen.display(String)
   * </code>.    
   *
   * @param s the screen acting as <code>Subject</code>
   */

  protected pointcut subjectChange(Subject s): 
    call(void Screen.display(String)) && target(s);

  /**
   * Defines how each <code>Observer</code> is to be updated when a change
   * to a <code>Subject</code> occurs. 
   *
   * @param s the subject on which a change of interest occured
   * @param o the observer to be notifed of the change  
   */
   
  protected void updateObserver(Subject s, Observer o) {
      ((Screen)o).display("Screen updated because screen displayed.");
  }  
}

