package examples.proxy.aspectj; 

/* -*- Mode: Java; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * This file is part of the design patterns project at UBC
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * either http://www.mozilla.org/MPL/ or http://aspectj.org/MPL/.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is ca.ubc.cs.spl.patterns.
 *
 * Contributor(s):   
 */

import ca.ubc.cs.spl.pattern.library.ProxyProtocol;
import org.aspectj.lang.JoinPoint;

/**
 * Implements a concrete proxy pattern instance. Here, all method calls from 
 * <code>Main</code> to <code>RealSubject.print(String)</code> are blocked.<p> 
 *
 * @author  Jan Hannemann
 * @author  Gregor Kiczales
 * @version 1.0, 05/13/02
 * 
 */  
 
public aspect DelegationProxy extends ProxyProtocol { 
    
    /** 
     * Assigns the <i>Subject</i> role to <code>RealSubject</code>.
     */
    
    declare parents: RealSubject implements Subject;

    /**
     * Captures all accesses to the subject that should be protected by 
     * this pattern instance. Here: All calls to <code>RealSubject.write(..)
     * </code>.
     */

	protected pointcut protectedAccess(): call(* RealSubject.write(..));


	/**
	 * Checks whether the access should be denied or not. Here: All accesses
	 * matched by the <code>protectedAccesses()</code> joinpoint.
	 *
     * @param caller the object responsible for the protected access
     * @param subject the subject receiving the call  
     * @param joinPoint the joinpoint associated with the protected access
     *
     * @returns true 
     */

	protected boolean rejection(Object caller, 
	                            Subject subject, 
	                            JoinPoint joinPoint) {
	    System.out.println("[DelegationProxy] delegating a write() call to "
	        +"AnotherRealSubject"); 
		return true;
	}

	/**
	 * For delegation: Provides an alternative return value if access 
	 * is denied. Here, it also calls an appropriate method on a delegate 
	 * (to illustrate how delegation works).
	 *
     * @param caller the object responsible for the protected access
     * @param subject the subject receiving the call  
     * @param joinPoint the joinpoint associated with the protected access
     *
     * @returns null
     */

	protected Object handleFailedAccess(Object caller, 
	                                    Subject subject, 
	                                    JoinPoint joinPoint) {
	    Object[] args = joinPoint.getArgs();
	    if (args != null) {
	        AnotherRealSubject.write((String)args[0]);
	    } else {
	        AnotherRealSubject.write("");
	    }
	    return null;
	}
}
