package examples.singleton.aspectj;

/* -*- Mode: Java; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * This file is part of the design patterns project at UBC
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * either http://www.mozilla.org/MPL/ or http://aspectj.org/MPL/.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is ca.ubc.cs.spl.patterns.
 *
 * Contributor(s):   
 */

/**
 * Implements the driver for the Singleton design pattern example.<p> 
 *
 * Intent: <i>Ensure that a class has only one instance and provide a global
 * point of access to it.</i><p>
 *
 * Participatng objects are <code>Printer</code> p1, p2, p3 and 
 * <code>PrinterSubclass</code> ps1, ps2, ps3.<p>
 *
 * Three different objects of both Printer and PrinterSubclass are
 * instantiated and compared. 
 *
 * This Implementation treats the singleton property as a non-inherited 
 * property. This meant that <i>Singleton</i> classes can still be subclassed
 * and these subclasses can access the <i>Singleton</i> constructor normally.
 *
 * <p><i>This is the AspectJ version.</i><p>
 *
 * @author  Jan Hannemann
 * @author  Gregor Kiczales
 * @version 1.0, 05/13/02
 * 
 * @see Printer
 * @see PrinterSubclass
 */

public class Main {

    /**
     * the three object references to instances of the <i>Singleton</i> class.
     */

	private static Printer         p1, p2, p3;
	
	// Experimental setup: Main creates three Printer objects.
	// The Printer implementation gives each object a unique ID
	// which is printed when print() is called. If the Singleton
	// implementation works, all three objects should be the same.
	//  
	// Implementation: AOP5 - One (concrete) aspect defines the behavior
	// of the pattern, creating a generic getInstance() method that
	// is attached to the Singleton interface. Another aspect assigns
	// the role to a particular class.
	//
	// The general description of the pattern (lthough not in an 
	// abstract aspect) is reusable.
	//
	// Considers different signatures for the constructor.
	//
	// Clients don't have to type-cast, they just use new(..)
	//
	// Subclasses are automatically Singletons, too, unless 
	// explicitly declared as non-singletons
	 
	
    /**
     * Implements the first test case. Creates 3 references to the 
     * <i>Singleton</i> by using the regular constructor. That should
     * create three identical objects.
     */	
	
	private static void test1() {
		System.out.println("Test 1: All three printers need to have the "
		 + "same ID");
		p1 = new Printer();
		p2 = new Printer();
		p3 = new Printer(); 
		p1.print();
		p2.print();
		p3.print(); 
	}
	


    /**
     * Implements the first test case. Tests if the 3 objects from test 1 are
     * in fact identical
     */	

	private static void test2() {
		System.out.println("Test 2: All three printers need to be identical");
		System.out.print("\tThey are ");
		if ((p1 == p2) && (p1 == p3)) { System.out.println("identical"); }
		else {System.out.println("not identical"); }
	}
	

    /**
     * Implements the first test case. Creates 3 instances of a <i>Singleton
     * </i>'s subclass. These objects should be different.
     */	

	private static void test3() {
		System.out.println("Test 3: Ensuring that subclasses can access the" 
		 + "constructor");
		System.out.println("        (All three outputs should be different)");
		p1 = new PrinterSubclass();
		p2 = new PrinterSubclass();
		p3 = new PrinterSubclass(); 
		p1.print();
		p2.print();
		p3.print(); 
	}
	
	
    /**
     * This is the driver for the <code>Singleton</code> case. It performes
     * three tests:
     *
     * <OL>
     *  <LI> Create 3 references to the <i>Singleton</i> by using the
     *       regular constructor. That should create three identical objects.
     *  <LI> Test if the above 3 objects are in fact identical
     *  <LI> Create 3 instances of a <i>Singleton</i>'s subclass. These
     *       objects should be different.
     * </OL>
     */
					
	public static void main (String[] args)
	{
		System.out.println("Testing pattern SINGLETON (aspectj) ...");
		test1();
		test2();
		test3();
		System.out.println("... done.");
	}
}