package examples.strategy.aspectj;


/* -*- Mode: Java; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 *
 * This file is part of the design patterns project at UBC
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * either http://www.mozilla.org/MPL/ or http://aspectj.org/MPL/.
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is ca.ubc.cs.spl.patterns.
 *
 * Contributor(s):   
 */

import ca.ubc.cs.spl.pattern.library.StrategyProtocol;
 
/**
 * Implements the the abstract strategy design pattern protocol<p>
 *
 * Intent: <i>Define a family of algorithms, encapsulate each one, and make
 * them interchangeable. Strategy lets the algorithm vary independently from 
 * clients that use it.</i><p>
 *
 * Participatng objects are <code>LinearSort</code> and <i>BubbleSort</i>
 * as <i>Strategies</i>, and <code>Sorter</code> as <i>Context</i>.
 *
 * In this example, an array of 10 numbers is to be sorted. Depending on the
 * number of arguments of the call to <code>Main</code>, linear sort or 
 * bubblesort are used as sorting algorithms. 
 *
 * <p><i>This is the AspectJ version.</i><p>
 *
 * @author  Jan Hannemann
 * @author  Gregor Kiczales
 * @version 1.0, 05/13/02
 * 
 * @see LinearSort
 * @see BubbleSort
 */


public aspect SortingStrategy extends StrategyProtocol {
    
	
    /**
     * Assignes the <i>Context</i> role to Sorter
     */

	declare parents: Sorter     implements Context;

    /**
     * Assignes the <i>Strategy</i> role to LinearSort
     */

	declare parents: LinearSort implements Strategy;

    /**
     * Assignes the <i>Strategy</i> role to BubbleSort
     */

	declare parents: BubbleSort implements Strategy;   
	
	
    /**
     * Invokes the appropriate strategy's sort() method when the
     * Sorter needs to sort.
     */

	int[] around(Sorter s, int[] numbers): call(int[] Sorter.sort(int[])) && target(s) && args(numbers) {
		Strategy strategy = getConcreteStrategy(s);
		if (strategy instanceof BubbleSort)
			((BubbleSort)strategy).sort(numbers);
		else if (strategy instanceof LinearSort)
			((LinearSort) strategy).sort(numbers);
		return numbers;
	}
}
